<#

WS-Backup-Win-11-USB-F.ps1
Workstation backup Windows 11 to USB F: drive

2025-05-30
yyyy-mm-dd

henrynet.org
Developed in USA

This PowerShell script backs up user files (signed in user only) from Windows 11
and Windows 10 computers to a USB drive or mapped network drive. The backup storage
device must be drive letter F: for successful operation. Any storage device mapped
as drive letter F: should work if the user has write access. Due to availability
and low cost, USB flash drive is assumed.

Before proceeding, a test is performed to validate drive F: storage. If drive F:
is writable, press <Enter> to continue. If drive F: is not writable, close the
window (click the X) and connect a storage device. Then try again.

The script attempts to record stored Wi-Fi profiles (network names) and the
associated passwords. If no profile exists, the following message would appear in
the PowerShell console:

"WARNING: Network connectivity may not be available, unable to reach remote sources."

The Wi-Fi password will be recorded in the machine information report only if the
account running the script is a member of the local "Administrators" group.

The script was developed to assist recovery efforts following a catastrophic event.

Tested successfully on these devices, both in a WORKGROUP:

[] Dell OptiPlex (manufactured 2025) Windows 11 Pro
[] Lenovo ThinkPad (manufactured 2022) Windows 10 Pro

It was not tested in domain environments.

Please note robocopy.exe operates at the folder level, not file level. Folders are
specified in the backup commands - not individual files. If script modification is
desired, do not specify individual files with robocopy commands. Robocopy would fail
with this message:

"ERROR <error code> Accessing Source Directory
The directory name is invalid."

The error would list the offending file name. A file is not a directory, hence the
error.

The following default user folders will be backed up:

Contacts
Desktop
Documents
Downloads
Favorites
Links
Music
OneDrive
Pictures
Videos

Browser bookmarks will be backed up for the following popular web browsers.

Microsoft Edge "Bookmarks" file in this folder:
C:\Users\%username%\AppData\Local\Microsoft\Edge\User Data\Default\

Google Chrome "Bookmarks" file in this folder:
C:\Users\%username%\AppData\Local\Google\Chrome\User Data\Default\

Brave "Bookmarks" file in this folder:
C:\Users\%username%\AppData\Local\BraveSoftware\Brave-Browser\User Data\Default\

Browser bookmarks are stored in a single file named "Bookmarks" (no file extension).

A robocopy log is created here:
F:\_WS_Backup_<hostname>_<username>\_Robocopy_Log\yyyy-MM-dd-T<hh>-<mm>-robocopy-log.txt

where <hostname> and <username> are the computer name, and signed in user. The log file
name contains the date and time. The "T" indicates time in 24-hour format.

Example, full path:
F:\_WS_Backup_Laptop_Mary\_Robocopy_Log\2025-05-26-T13-32-robocopy-log.txt

hostname: Laptop
username: Mary
date:     2025-05-26
time:     1:32 pm

Robocopy sometimes hides newly created folders. Use the "attrib"
command to unhide a folder after backup operations are finished
if this occurs.

A report is created on machine information and stored in the "\Machine" folder.
It contains these details:

[] Make and model of computer
[] Serial number (Dell service tag)
[] Operating system name
[] Hostname
[] Domain name
[] Username
[] Wi-Fi network name and password
[] List of installed applications
[] Storage devices
[] MAC address of physical network adapters

The list of installed applications in the machine report is nearly the same as
Programs and Features in Control Panel.

The "Recycle_Bin" folder is for future use. Backup of the Recycle Bin can be done
manually if desired.

How to run the script:
Right-click the file and select "Run with PowerShell"

#>

# Start script


# Set global error handling to suppress red text errors
$ErrorActionPreference = 'SilentlyContinue'

# Set title bar label
$host.ui.RawUI.WindowTitle = "Validate Storage Device"

# Insert blank line in console
Write-Host ""

   $driveLetter = "F:"
   $testFile = "$driveLetter\test.txt"

   try {
       # Create a temporary file
       New-Item -ItemType File -Path $testFile -Value "test" -Force | Out-Null

        # Check if the file was created successfully
        if (Test-Path $testFile) {
            Write-Host "Drive $driveLetter is writable. Ok to proceed."
           # Delete the temporary file
            Remove-Item $testFile -Force
        } else {
             Write-Host "Drive $driveLetter is not writable, cannot proceed."
        }

   } catch {
       Write-Host "Drive $driveLetter is not writable. Error: $($_.Exception.Message)"
   }

# Insert blank line in console
Write-Host ""

Read-Host -Prompt "Press Enter if Drive F: is writable.
Close if Drive F: is not writable."


# Insert blank line in console
Write-Host ""


# Get date and store as variable
$date = Get-Date -Format "yyyy-MM-dd"

# Get time and store as variable
$currentTimeOnly = Get-Date -Format "HH-mm"

# Store hostname and username as one variable
$host_User = "$($env:COMPUTERNAME)_$($env:USERNAME)"

# Create folder in root of USB drive based on hostname and username
$folderPath = "F:\_WS_Backup_$host_User"
if (!(Test-Path -Path $folderPath)) {
  New-Item -ItemType Directory -Path $folderPath
}


# Create folders if they do not exist


# Contacts
if (!(Test-Path -Path $folderPath\Contacts)) {
  New-Item -ItemType Directory -Path $folderPath\Contacts
}


# Desktop
if (!(Test-Path -Path $folderPath\Desktop)) {
  New-Item -ItemType Directory -Path $folderPath\Desktop
}


# Documents
if (!(Test-Path -Path $folderPath\Documents)) {
  New-Item -ItemType Directory -Path $folderPath\Documents
}


# Downloads
if (!(Test-Path -Path $folderPath\Downloads)) {
  New-Item -ItemType Directory -Path $folderPath\Downloads
}


# Favorites
if (!(Test-Path -Path $folderPath\Favorites)) {
  New-Item -ItemType Directory -Path $folderPath\Favorites
}


# Links
if (!(Test-Path -Path $folderPath\Links)) {
  New-Item -ItemType Directory -Path $folderPath\Links
}


# Music
if (!(Test-Path -Path $folderPath\Music)) {
  New-Item -ItemType Directory -Path $folderPath\Music
}


# OneDrive
if (!(Test-Path -Path $folderPath\OneDrive)) {
  New-Item -ItemType Directory -Path $folderPath\OneDrive
}


# Pictures
if (!(Test-Path -Path $folderPath\Pictures)) {
  New-Item -ItemType Directory -Path $folderPath\Pictures
}


# Videos
if (!(Test-Path -Path $folderPath\Videos)) {
  New-Item -ItemType Directory -Path $folderPath\Videos
}


# Recycle Bin
if (!(Test-Path -Path $folderPath\Recycle_Bin)) {
  New-Item -ItemType Directory -Path $folderPath\Recycle_Bin
}


# Robocopy Log
if (!(Test-Path -Path $folderPath\_Robocopy_Log)) {
  New-Item -ItemType Directory -Path $folderPath\_Robocopy_Log
}


# Machine
if (!(Test-Path -Path $folderPath\Machine)) {
  New-Item -ItemType Directory -Path $folderPath\Machine
}


# Edge bookmarks
if (!(Test-Path -Path $folderPath\Edge-bookmarks)) {
  New-Item -ItemType Directory -Path $folderPath\Edge-bookmarks
}


# Brave bookmarks
if (!(Test-Path -Path $folderPath\Brave-bookmarks)) {
  New-Item -ItemType Directory -Path $folderPath\Brave-bookmarks
}


# Chrome bookmarks
if (!(Test-Path -Path $folderPath\Chrome-bookmarks)) {
  New-Item -ItemType Directory -Path $folderPath\Chrome-bookmarks
}


# Start backup operations


# Set title bar label
$host.ui.RawUI.WindowTitle = "Contacts - backup in progress"
robocopy C:\Users\$env:USERNAME\Contacts $folderPath\Contacts /E /R:0 /W:0 /TEE /NP /XJ /LOG+:$folderPath\_Robocopy_Log\$date-T$currentTimeOnly-robocopy-log.txt

Start-Sleep -Seconds 5

# Set title bar label
$host.ui.RawUI.WindowTitle = "Desktop - backup in progress"
robocopy C:\Users\$env:USERNAME\Desktop $folderPath\Desktop /E /R:0 /W:0 /TEE /NP /XJ /LOG+:$folderPath\_Robocopy_Log\$date-T$currentTimeOnly-robocopy-log.txt

Start-Sleep -Seconds 5

# Set title bar label
$host.ui.RawUI.WindowTitle = "Documents - backup in progress"
robocopy C:\Users\$env:USERNAME\Documents $folderPath\Documents /E /R:0 /W:0 /TEE /NP /XJ /LOG+:$folderPath\_Robocopy_Log\$date-T$currentTimeOnly-robocopy-log.txt

Start-Sleep -Seconds 5

# Set title bar label
$host.ui.RawUI.WindowTitle = "Downloads - backup in progress"
robocopy C:\Users\$env:USERNAME\Downloads $folderPath\Downloads /E /R:0 /W:0 /TEE /NP /XJ /LOG+:$folderPath\_Robocopy_Log\$date-T$currentTimeOnly-robocopy-log.txt

Start-Sleep -Seconds 5

# Set title bar label
$host.ui.RawUI.WindowTitle = "Favorites - backup in progress"
robocopy C:\Users\$env:USERNAME\Favorites $folderPath\Favorites /E /R:0 /W:0 /TEE /NP /XJ /LOG+:$folderPath\_Robocopy_Log\$date-T$currentTimeOnly-robocopy-log.txt

Start-Sleep -Seconds 5

# Set title bar label
$host.ui.RawUI.WindowTitle = "Links - backup in progress"
robocopy C:\Users\$env:USERNAME\Links $folderPath\Links /E /R:0 /W:0 /TEE /NP /XJ /LOG+:$folderPath\_Robocopy_Log\$date-T$currentTimeOnly-robocopy-log.txt

Start-Sleep -Seconds 5

# Set title bar label
$host.ui.RawUI.WindowTitle = "Music - backup in progress"
robocopy C:\Users\$env:USERNAME\Music $folderPath\Music /E /R:0 /W:0 /TEE /NP /XJ /LOG+:$folderPath\_Robocopy_Log\$date-T$currentTimeOnly-robocopy-log.txt

Start-Sleep -Seconds 5

# Set title bar label
$host.ui.RawUI.WindowTitle = "OneDrive - backup in progress"
robocopy C:\Users\$env:USERNAME\OneDrive $folderPath\OneDrive /E /R:0 /W:0 /TEE /NP /XJ /LOG+:$folderPath\_Robocopy_Log\$date-T$currentTimeOnly-robocopy-log.txt

Start-Sleep -Seconds 5

# Set title bar label
$host.ui.RawUI.WindowTitle = "Pictures - backup in progress"
robocopy C:\Users\$env:USERNAME\Pictures $folderPath\Pictures /E /R:0 /W:0 /TEE /NP /XJ /LOG+:$folderPath\_Robocopy_Log\$date-T$currentTimeOnly-robocopy-log.txt

Start-Sleep -Seconds 5

# Set title bar label
$host.ui.RawUI.WindowTitle = "Videos - backup in progress"
robocopy C:\Users\$env:USERNAME\Videos $folderPath\Videos /E /R:0 /W:0 /TEE /NP /XJ /LOG+:$folderPath\_Robocopy_Log\$date-T$currentTimeOnly-robocopy-log.txt

Start-Sleep -Seconds 5

# Edge bookmarks backup
Copy-Item -Path "C:\Users\$env:USERNAME\AppData\Local\Microsoft\Edge\User Data\Default\Bookmarks" -Destination "$folderPath\Edge-bookmarks"

# Brave bookmarks backup
Copy-Item -Path "C:\Users\$env:USERNAME\AppData\Local\BraveSoftware\Brave-Browser\User Data\Default\Bookmarks" -Destination "$folderPath\Brave-bookmarks"

# Chrome bookmarks backup
Copy-Item -Path "C:\Users\$env:USERNAME\AppData\Local\Google\Chrome\User Data\Default\Bookmarks" -Destination "$folderPath\Chrome-bookmarks"


# Create machine information report


# Put title at top of report
"Machine Information Report" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt

# Put date and time at top of report (4 spaces between date and time)
Get-Date -Format "yyyy-MM-dd    " | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -NoNewLine -append
Get-Date -Format "hh:mm:ss tt" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
"via PowerShell script" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -NoNewLine -append

# Insert blank lines in log file
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Put heading in log file
"Make & model:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -NoNewLine -append
Get-CimInstance -ClassName Win32_ComputerSystem | Select-Object -Property Manufacturer, Model | Format-Table -HideTableHeaders | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Put heading in log file
"Serial number:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -NoNewLine -append
Get-CimInstance win32_bios | select SerialNumber | Format-Table -HideTableHeaders | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Get operating system
systeminfo | findstr /B /C:"OS Name:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Get OS sub-version (e.g. 24H2)
Get-ItemPropertyValue 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion' DisplayVersion | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Insert blank lines in log file
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Put heading in log file
"Hostname:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
[System.Net.Dns]::GetHostByName($env:computerName).HostName | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Insert blank lines in log file
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Put heading in log file
"Domain name:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -NoNewLine -append
Get-CimInstance -ClassName Win32_ComputerSystem | Select-Object -Property Domain | Format-Table -HideTableHeaders | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Put heading in log file
"Username:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
$env:username | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Insert blank lines in log file
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Put heading in log file
"Wi-Fi:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
netsh wlan show profile * key=clear | findstr "name" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Insert blank lines in log file
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Put heading in log file
"Wi-Fi password:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
netsh wlan show profile * key=clear | findstr "Content" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Insert blank lines in log file
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
"" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Put heading in log file
"Installed applications:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
Get-Package -ProviderName MSI, Programs | Sort-Object -Property Name | Format-Table -AutoSize -wrap Name,Version | Out-String -width 128 | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
Start-Sleep -second 3

# Insert blank line in console
""

# Put heading in log file
"Storage devices:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
Get-PhysicalDisk | Select-Object DeviceId, FriendlyName, HealthStatus | Format-Table -AutoSize | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Put heading in log file
"MAC address, physical network adapters:" | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append
Get-NetAdapter | Select-Object MacAddress | Format-Table -HideTableHeaders -AutoSize | Out-File $folderPath\Machine\$date-T$currentTimeOnly-machine-information.txt -append

# Insert blank line in console
""

# Set title bar label
$host.ui.RawUI.WindowTitle = "Backup operations finished"

Write-Host "End"
Write-Host $date

# Pause script
pause

# Open file explorer
explorer F:\
